set Release(edit.tcl) {$Header: /home/cvs/tktest/edit.tcl,v 1.12 2004/10/05 01:17:35 clif Exp $}

###################################################################
# FormatEvent: format the event for showing in the list of events.
###################################################################
proc FormatEvent {delay subscript replaceList} {
    #
    # USER OPTION: you can modify this procedure to change the way events
    # are displayed in the script list box.
    #
    if {[regexp {Bind,(.*),<([^>]*)>} $subscript all p1 p2]} {
        #
        # For ordinary events (they begin with "Bind,"),
        # show the key pressed for Key events
        #
        if {$p2 == "Key"} {
            set key [format {-%s} [rrlookup A $replaceList]]
        } else {
            set key ""
        }
        set entry [format {%4.1f <%s%s>   %s} [expr $delay/10.0] $p2 $key $p1]
    } else {
        #
        # These are special events.
        # For Comments and TclExec's show a little bit of the
        # comment/code.
        #
        switch -- $subscript \
            BeginComment - \
            ExecTcl {
                set text [lindex $replaceList 0]
                # replace newlines with something
                while 1 {
                    set n [string first \n $text]
                    if {$n < 0} {
                        break
                    }
                    switch -- $subscript \
                        ExecTcl {
                            set nl ";"
                        } \
                        BeginComment - \
                        default {
                            set nl "\\n"
                        }
                    set text [format {%s%s%s} \
                        [string range $text 0 [expr $n-1]] \
                        $nl \
                        [string range $text [expr $n+1] end]]
                }
                set len [string length $text]
                if {$len <= 38} {
                    set detail [format { "%s"} $text]
                } else {
                    set detail [format { "%s...%s"} \
                        [string range $text 0 24] \
                        [string range $text [expr $len-11] end]]
                }
            } \
            ExecScript {
                set detail " $replaceList"
            } \
            default {
                set detail ""
            }
        set entry [format {%4.1f %s%s} [expr $delay/10.0] $subscript $detail]
    }
    return $entry
}
###################################################################
# InsertAction: insert an action in the replay script.
#     delay: the time between this event and the last event
#     app: the application that the action occurred in
#     subscript: the subscript of the event which records everything
#         we need to know about it, in particular, where to find
#         the script of the original binding
#     replaceList: the details of the event so we can replay it exactly
#          as it happened.
###################################################################
proc InsertAction {delay app subscript replaceList} {
    global ReplayData 
    global GUI
    set level $ReplayData(Level)
    set item [list $delay $app $subscript $replaceList]
    set index $ReplayData($level,ScriptIndex)
    # do not allow insertion before the "beginning of script" item
    if {$index == 0} {
        if {[string compare [lindex $item 2] {--- Beginning of script ---}] \
                != 0} {
            set index 1
        }
    }
    set ReplayData($level,Script) \
        [linsert $ReplayData($level,Script) $index $item]
    $GUI(eventlistbox) insert $index \
        [FormatEvent $delay $subscript $replaceList]
    incr ReplayData($level,ScriptIndex)
    SelectActionListItem $ReplayData($level,ScriptIndex) see
}
###################################################################
# CompressMotionEvents:  the procedures goes through the current script
#   and removes all mouse "Motion" events that are followed by another
#   Motion event.  The idea is that the intermediate Motion events are
#   not important, only the final destination.  TkReplay usually (it
#   is an option) moves smoothly between locations anyway.
#   If the intermediate Motion events *are* important than you should
#   not use this procedure.
###################################################################
proc CompressMotionEvents {} {
    global ReplayData
    set level $ReplayData(Level)
    MsgToUser "Compressing events in script.  Please wait ..."
    set oldCursor [ChangeCursor watch]
    set newScript {}
    set lastItem ""
    set lastEvent NotMotion
    set lastWhich ""
    set which ""
    foreach thisItem $ReplayData($level,Script) {
        set thisEvent NotMotion
        if {[regexp {Bind,([^,]*),(.*),*(<[^>]+>)} \
                [lindex $thisItem 2] what which junk event]} {
            if {[string first Motion $event] >= 0} {
                set thisEvent Motion
            }
        }
	# lastWhich test is to filter out events bound to canvas
	#  when compressing canvas element moves -
	#  For Example: if B3 is bound to an item to drag, it
	#  generates a which=Canvas event for <B3-Motion>
	#  and a which=.cvsName event for <Motion>
        #    Only the <B3-Motion> event is pertinent.
	#  but the original logic retained the <Motion> event instead.
	if {($lastWhich eq "Canvas")  &&
	    ($thisEvent eq "Motion")} {
	    set lastWhich $which
	    continue
	}
	  

        if {($lastItem!="") && \
                (($lastEvent!="Motion") || ($thisEvent!="Motion")) } {
            lappend newScript $lastItem
        }
        set lastItem $thisItem
        set lastEvent $thisEvent
	set lastWhich $which
    }
    lappend newScript $lastItem
    # reconstruct the list of events
    EraseScript all
    foreach item $newScript {
        eval InsertAction $item
    }
    set ReplayData($level,ScriptIndex) 0
    SelectActionListItem 0 see
    ChangeCursor $oldCursor
    MsgToUser "Motion events have been compressed."
}
###################################################################
# MoveEvent: move an event around in the event list.
#   First it does some checking to prevent removing of the end markers.
#   Then it does the move by deleting the event and inserting it
#   somewhere else.
###################################################################
proc MoveEvent {places} {
    global ReplayData
    set level $ReplayData(Level)
    set index $ReplayData($level,ScriptIndex)
    set newPlace [expr $index + $places]
    set max [expr [llength $ReplayData($level,Script)] - 2]
    #
    # Do various error checking to make sure the script remains valid.
    #
    if {$index == 0} {
        MsgToUser "You cannot move the beginning-of-script marker" low
        return
    }
    if {$newPlace < 1} {
        MsgToUser "You cannot move anything before the beginning-of-script\
            marker" low
        return
    }
    if {$index > $max} {
        MsgToUser "You cannot move the end-of-script marker" low
        return
    }
    if {$newPlace > $max} {
        MsgToUser "You cannot move anything after the end-of-script\
            marker" low
        return
    }
    #
    # Move by deleting and inserting in the new place.
    #
    set action [lindex $ReplayData($level,Script) $index]
    DeleteEvent
    incr ReplayData($level,ScriptIndex) $places
    eval InsertAction $action
    incr ReplayData($level,ScriptIndex) -1
    SelectActionListItem $ReplayData($level,ScriptIndex) see
    MsgToUser "One script action was moved" info
}
###################################################################
# DeleteEvent: removes an event from the event list.
###################################################################
proc DeleteEvent {} {
    global ReplayData
    global GUI
    set level $ReplayData(Level)
    set index $ReplayData($level,ScriptIndex)
    set max [expr [llength $ReplayData($level,Script)] - 2]
parray ReplayData
    #
    # Error checking
    #
    if {$index == 0} {
        MsgToUser "You cannot delete the beginning-of-script marker" low
        return
    }
    if {$index > $max} {
        MsgToUser "You cannot delete the end-of-script marker" low
        return
    }
    set ReplayData($level,Script) \
        [lreplace $ReplayData($level,Script) $index $index]
    $GUI(eventlistbox) delete $index
    SelectActionListItem $ReplayData($level,ScriptIndex) see
    MsgToUser "One script action was deleted" info
}
###################################################################
# EditEvent: allows you to change the delay of an event.
###################################################################
proc EditEvent {} {
    global ReplayData
    set level $ReplayData(Level)
    set w .editBox
    if [winfo exists $w] {
        wm deiconify $w
    } else {
        toplevel $w
        wm title $w "Edit event delay"
        message $w.l1 -text "" -aspect 450
        pack $w.l1 -side top
        frame $w.fr
        pack $w.fr -side top
        label $w.fr.label -text "New delay:"
        pack $w.fr.label -side left
        entry $w.fr.entry -bd 2 -relief sunken
        pack $w.fr.entry -side left -fill x -expand yes
        bind $w.fr.entry <Return> "ChangeDelay $w"
        frame $w.buttons
        pack $w.buttons -side top
        button $w.buttons.apply -text "Apply" -command "
            ChangeDelay $w
            wm withdraw $w
        "
        pack $w.buttons.apply -side left
        button $w.buttons.cancel -text "Cancel" -command "wm withdraw $w"
        pack $w.buttons.cancel -side left
    }
    set index $ReplayData($level,ScriptIndex)
    set action [lindex $ReplayData($level,Script) $index]
    set delay [lindex $action 0]
    $w.l1 configure -text [format {Current delay for this event is %.1f.\
Type in a new value and press Apply or the Return key.} [expr $delay/10.0]]
}
###################################################################
# ChangeDelay -- callback for EditEvent.
###################################################################
proc ChangeDelay {w} {
    global ReplayData
    set level $ReplayData(Level)
    set index $ReplayData($level,ScriptIndex)
    set action [lindex $ReplayData($level,Script) $index]
    set newDelay [expr int([$w.fr.entry get] * 10)]
    set newAction [lreplace $action 0 0 $newDelay]
    DeleteEvent
    InsertAction $newDelay [lindex $action 1] [lindex $action 2] \
        [lindex $action 3]
    wm withdraw $w
    MsgToUser "Delay was changed" info
}
###################################################################
# Rewind: makes the first event the current event.
###################################################################
proc Rewind {} {
    global ReplayData
    global GUI

    set level $ReplayData(Level)
    set ReplayData($level,ScriptIndex) 0
    $GUI(eventlistbox) yview 0
    SelectActionListItem 0 see
    MsgToUser "Script rewound to beginning" info
}
###################################################################
# AddPause: adds a Pause event.
###################################################################
proc AddPause {} {
    global ReplayData
    InsertAction 0 ThisApp Pause ""
}
###################################################################
# EraseScript: erases all the events in the event list.
###################################################################
proc EraseScript {{how inside}} {
    global ReplayData
    global GUI

    set level $ReplayData(Level)
    MsgToUser "Erase the entire script"
    if {$how != "leveldown"} {
        set ReplayData($level,Script) {}
    }
    $GUI(eventlistbox) delete 0 end
    set ReplayData(ScriptFileName) {}

    if {$how == "inside" || $how == "leveldown"} {
        set ReplayData($level,ScriptIndex) 0
        InsertAction 0 ThisApp {--- End of script ---} ""
        set ReplayData($level,ScriptIndex) 0
        InsertAction 0 ThisApp {--- Beginning of script ---} ""
    }
}
###################################################################
# StartRecording
###################################################################
proc StartRecording {} {
    global ReplayData GUI
    MsgToUser "Start recording a script"
    set ReplayData(RecordingOn) 1
    set ReplayData(LastEventAt) 0
    set ReplayData(Status) Recording

    #
    # enable commands that are only allowed while recording
    #

#    foreach label {"Exec Script..." "Add Comment" "Insert Tcl" "Add Pause"} {
#        .replay.mm.record entryconfigure $label -state normal
#    }

    foreach win [winfo children $GUI(bbox2)] {
      $win configure -state normal
    }


    foreach win [winfo children $GUI(editFrame)] {
      $win configure -state normal
    }

    # turn on the timer
    set ReplayData(Timer) 0
    ReplayTimerTick
}
###################################################################
# StopRecording
###################################################################
proc StopRecording {} {
    global ReplayData
    global GUI
    MsgToUser "Stop recording a script"
    set ReplayData(RecordingOn) 0
    set ReplayData(PlayingOn) 0
    set ReplayData(Status) Connected

    #
    # disable commands that are only allowed while recording
    #

    foreach win [winfo children $GUI(editFrame)] {
      $win configure -state disabled
    }

    foreach win [winfo children $GUI(editFrame)] {
      $win configure -state disabled
    }

    # turn off the timer
    set ReplayData(Timer) -1
}
###################################################################
# Replay
###################################################################
proc Replay {} {
    global ReplayData
    MsgToUser "Start replaying the script"
    set ReplayData(PlayingOn) 1
    set ReplayData(UseDelays) 1
    set ReplayData(Status) Playing
    ReplayActions
    set ReplayData(PlayingOn) 0
    set ReplayData(Status) Connected
    MsgToUser "Finished replaying the script"
}
###################################################################
# ReplayFast: replays the events with no time delays.
###################################################################
proc ReplayFast {} {
    global ReplayData
    MsgToUser "Start replaying a script (no delays)"
    set ReplayData(PlayingOn) 1
    set ReplayData(UseDelays) 0
    set ReplayData(Status) Playing
    ReplayActions
    set ReplayData(PlayingOn) 0
    set ReplayData(Status) Connected
    MsgToUser "Finished replaying the script"
}
###################################################################
# Save: saves the event list in the file.
###################################################################
proc Save {} {
    global ReplayData
    set level $ReplayData(Level)
    MsgToUser "Saving file. Please wait ..." info
    WriteFile $ReplayData($level,ScriptFileName)
}
###################################################################
# WriteFile: writes the script list to a file.
###################################################################
proc WriteFile {filename} {
    global ReplayData
    set level $ReplayData(Level)
    set fid [open $filename w]
    # remove the dummy begin and end events
    set rdScript $ReplayData($level,Script)
    set script [lrange $rdScript 1 [expr [llength $rdScript]-2]]
    foreach event $script {
        puts $fid [format "InsertAction {%s} {%s} {%s} {%s}" \
            [lindex $event 0] [lindex $event 1] \
            [lindex $event 2] [lindex $event 3]]
    }
    close $fid
    MsgToUser "Saved script to \"[tildefy $filename]\""
}
###################################################################
# SaveAs: saves the event list with a new file name.
###################################################################
proc SaveAs {} {
    global ReplayData
    set level $ReplayData(Level)
    MsgToUser "Select a name to save the script to"

    set filename [tk_getSaveFile -defaultextension .tkr \
        -filetypes {{{TkReplay Files} .tkr} {Script .scr} {tcl {.tcl .tk}}} \
	-initialdir $ReplayData(scriptFileDir)]

    if {$filename == ""} {
        MsgToUser "Save cancelled"
        return
    }

    set ReplayData($level,ScriptFileName) $filename
    SetScriptFileName
    MsgToUser "Saving file as [tildefy $filename]. Please wait ..." info
    set oldCursor [ChangeCursor watch]
    WriteFile $filename
    ChangeCursor $oldCursor
}
###################################################################
# Load: inserts a file into the event list.
###################################################################
proc Load {} {
    global ReplayData GUI
    # MsgToUser "Select a file to load the script from"

    set filename [tk_getOpenFile -defaultextension .tkr \
        -filetypes {{{TkReplay Files} .tkr} {Script .scr} {tcl {.tcl .tk}}} \
	-initialdir $ReplayData(scriptFileDir)]
    if {$filename == ""} {
       MsgToUser "Load cancelled"
       return
    }
    if {[string equal .tkr [file extension $filename]]} {
      set ReplayData(scriptFileDir) [file dirname $filename]
      LoadScript $filename
    } else {
      LoadControl $filename
      $GUI(nb) raise control
  }
}
###################################################################
# LoadScript
###################################################################
proc LoadScript {filename} {
    global ReplayData
    set level $ReplayData(Level)
    MsgToUser "Loading script \"[tildefy $filename]\".  Please wait ..."
    set oldCursor [ChangeCursor watch]
    if {[file readable $filename]} {
        EraseScript
        source $filename
        Rewind
        set ReplayData($level,ScriptFileName) $filename
        SetScriptFileName
        MsgToUser "Loaded script \"[tildefy $filename]\"" info
        set ret 1
    } else {
        MsgToUser "Cannot read \"[tildefy $filename]\", not loaded" medium
        set ret 0
    }
    ChangeCursor $oldCursor
    return $ret
}
###################################################################
# LoadAndConnect: load application and then connect to it.
###################################################################
proc LoadAndConnect {scriptName} {
    global ReplayData
    MsgToUser "Load and connect to \"$scriptName\""
    if {![lmember $scriptName [winfo interps]]} {
        set appCmd $scriptName
        if {[file executable [lindex $scriptName 0]] == 0} {
            # if the file is not executable, prepend the name of a wish
            set appCmd [concat $ReplayData(DefaultWish) $appCmd]
        }
        if {![LoadApp $appCmd]} {
            # if load fails do not try to connect
            return
        }
        # wait 2 seconds for it start up
        after 2000
    } else {
        puts "$scriptName already loaded"
    }
    # Don't use the full path to the script as the name of the app
    # Use just the last bit
    ConnectToApp [file tail [lindex $scriptName 0]]
}
###################################################################
# LoadApp: load an application and connect to it.
###################################################################
proc LoadApp {appCmd} {
    global ReplayData
    if {$ReplayData(RecordingOn)} {
        InsertAction 0 ThisApp LoadApp $appCmd
    }
    MsgToUser "Starting application \"$appCmd\".  Please wait ..."
    set oldCursor [ChangeCursor watch]
    if {[catch [list eval exec $appCmd &] ret]} {
        MsgToUser "Load of \"$appCmd\" failed" low
        set ret 0
    } else {
        MsgToUser "Load of \"$appCmd\" succeeded"
        set ret 1
    }
    ChangeCursor $oldCursor
    return $ret
}


proc RebuildScript {} {
    global ReplayData
    MsgToUser "Start replaying the script"
    set ReplayData(PlayingOn) 1
    set ReplayData(UseDelays) 1
    set ReplayData(Status) Playing
    RebuildActions
    set ReplayData(PlayingOn) 0
    set ReplayData(Status) Connected
    MsgToUser "Finished replaying the script"
}

